// @ts-check
const dgram = require("dgram");
const _ = require("underscore");
const events = require("events");
const Device = require("./device");

// Constants
const log = console.log;
const SERVER_PORT = 3000;
const SERVER_PORT_OLD = 3001;
const SERVER_PORT_VERY_OLD = 3011; // hat Eddy
const IGNORED_IPS = ["192.168.10.10"];

class NetworkManager extends events.EventEmitter {
    constructor() {
        super();
        this.devices = {};
        this.servers = [];
    }

    start() {
        _.each([SERVER_PORT, SERVER_PORT_OLD, SERVER_PORT_VERY_OLD], port => {
            // Start UDP Server
            const server = dgram.createSocket("udp4");
            server.on("error", err => {
                log(`UDP Server Error: ${err.stack}`);
                this.stop();
            });

            server.on("message", (msg, remote) => {
                // Id and addr generation (for test devices)
                let id = null;
                const now = Date.now();
                let addr = remote.address;
                if (addr && (!this.devices[addr] || (this.devices[addr] && this.devices[addr].destroyed))) {
                    // IP check
                    if (_.contains(IGNORED_IPS, addr)) {
                        console.log("Ignoring device with IP: " + addr);
                    } else if (validNetwork(addr)) {
                        // Create new device
                        this.devices[addr] = new Device(this, remote.address, 2000, id);
                        this.devices[addr].locked = true;
                        this.devices[addr].start();
                    } else {
                        console.log("Invalid IP: " + addr);
                        this.devices[addr] = new Device(this, remote.address, 2000, id);
                        this.devices[addr].state.ping = now;
                        this.emit("discover", {
                            id,
                            ip: addr,
                            properties: {
                                invalidIp: true
                            }
                        });
                    }
                } else if (addr && this.devices[addr] && this.devices[addr].state.ping > 0) {
                    this.devices[addr].state.ping = now;
                }
            });

            server.on("listening", () => {
                const address = server.address();
                log(`UDP Server listening on ${address.address}:${address.port}`);
            });

            server.bind(port);

            // Add to servers
            this.servers.push(server);
        });
    }

    stop() {
        log("Shutting down server");

        // Shutdown server
        _.each(this.servers, server => {
            server.close();
        });

        this.servers = [];
    }

    disconnectDevices() {
        _.each(this.devices, device => device.stop());
    }

    disconnect(moduleId, remove = false) {
        const device = _.find(this.devices, device => device.id == moduleId);
        if (device && device.address && remove) {
            delete this.devices[device.address];
        }

        if (device) {
            device.stop();
        }
    }

    /**
     * @param {number|string} moduleId
     * @param {string|any[]} commandId
     * @param {*} params
     */
    request(moduleId, commandId, params = null) {
        const device = _.find(this.devices, device => device.id == moduleId);
        if (device && !device.destroyed) {
            //  && !device.locked
            return device.write(commandId, params);
        } else if (device && device.destroyed) {
            console.warn(`Device ${moduleId} disconnected`);
            return Promise.resolve();
        } else if (device && device.locked) {
            console.warn(`Device ${moduleId} locked`);
            return Promise.resolve();
        } else {
            console.warn(`Device ${moduleId} not found`);
            return Promise.resolve();
        }
    }

    /**
     * @param {number|string} deviceType
     * @param {string} filename
     */
    requestScript(deviceType, filename) {
        const device = new Device(this, "192.168.10.10", 2000, "GENERATED_DEVICE");
        device.type = deviceType;
        device.start();
        return device.executeFile(filename);
    }

    /**
     * @param {number} moduleId
     * @param {*} message
     */
    response(moduleId, message) {
        // Only process responses when online
        this.emit(
            message.id,
            _.extend(message, {
                moduleId
            })
        );
    }
}

module.exports = new NetworkManager();
